// Copyright 2014 Google Inc. All Rights Reserved.

#ifndef ANDROID_AUTO_PROJECTION_PROTOCOL_PHONE_STATUS_ENDPOINT_H
#define ANDROID_AUTO_PROJECTION_PROTOCOL_PHONE_STATUS_ENDPOINT_H

#include "common.h"
#include "ProtocolEndpointBase.h"
#include "util/Mutex.h"
#include "IPhoneStatusCallbacks.h"

/**
 * This class implements an endpoint that receives phone status over
 * the GAL protocol. You must also subclass IPhoneStatusCallbacks and implement all
 * the methods so that the GAL receiver can interface with your hardware. The call
 * sequence would look something like this:
 * <br>
 * <pre>
 *      galReceiver->init();
 *      ... Initialization code ...
 *      PhoneStatusEndpoint* endpoint = new PhoneStatusEndpoint(serviceId,
 *              galReceiver->messageRouter());
 *      endpoint->registerCallbacks(callbacks); // Subclassed from IPhoneStatusCallbacks.
 *      galReceiver->registerService(endpoint);
 *      ... Other Initialization code ...
 *      galReceiver->start();
 * </pre>
 */
class PhoneStatusEndpoint : public ProtocolEndpointBase {
public:
    PhoneStatusEndpoint(uint8_t id, MessageRouter *router) :
            ProtocolEndpointBase(id, router, false) { }

    void addDiscoveryInfo(ServiceDiscoveryResponse *sdr);
    int routeMessage(uint8_t channelId, uint16_t type, const shared_ptr<IoBuffer>& msg);
    void registerCallbacks(const shared_ptr<IPhoneStatusCallbacks>& callbacks) {
        mCallbacks = callbacks;
    }

    /**
     * Reports an action associated with current phone status.
     * @param caller_number The number of the call to act upon. Can be empty.
     * @param caller_id The caller id to act upon. Can be empty.
     * @param action See enum InstrumentClusterAction in protos.proto for possible values.
     * @return STATUS_SUCCESS if the call succeeded, an error code otherwise.
     */
    int reportAction(const string& caller_number, const string& caller_id, uint32_t action);

private:
    shared_ptr<IPhoneStatusCallbacks> mCallbacks;
    void handlePhoneStatus(const PhoneStatus &status);
};

#endif // ANDROID_AUTO_PROJECTION_PROTOCOL_PHONE_STATUS_ENDPOINT_H
